	function bitspec = gfir(firspec, arithspec, genopts)
	%bitspec = gfir(firspec, arithspec[, genopts])
	%
	%Sets up parameters for partial product generation for FIR filters:
	%filter coefficients and correction term.
	%
	%Arguments:
	%  firspec - filter specification structure with fields
	%    numin: number of input branches
	%    numout: number of output branches
	%    h: cell array of the impulse responses for each output branch
	%  arithspec - arithmetics specification structure with fields
	%    wdata: data wordlength
	%    signedcoeffs: 1 if signed digit coefficients are used
	%    signeddata: 1 if the data is signed
	%    ctype: coefficient type ('bin' or 'msd')
	%    cid: coefficient index (for 'msd')
	%  genopts - generator options
	%    verbose: verbosity level (0, 1 or 2)
	%
	%Returns:
	%  bitspec - structure with fields
	%    numin: number of input branches
	%    numout: number of output branches
	%    branches: cell array of branch specification structures
	%    arith: arithmetic specification structure
	
	%Copyright 2008, 2010 Anton Blad
	%
	%This file is part of firgen.
	%
	%firgen is free software: you can redistribute it and/or modify
	%it under the terms of the GNU General Public License as published by
	%the Free Software Foundation, either version 3 of the License, or
	%(at your option) any later version.
	%
	%firgen is distributed in the hope that it will be useful,
	%but WITHOUT ANY WARRANTY; without even the implied warranty of
	%MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	%GNU General Public License for more details.
	%
	%You should have received a copy of the GNU General Public License
	%along with firgen.  If not, see <http://www.gnu.org/licenses/>
	
	if nargin < 3
		genopts = [];
	end
	genopts = parseopts(genopts);
	
	if ~isfield(arithspec, 'cid')
		arithspec.cid = 0;
	end
	
	bitspec.numin = firspec.numin;
	bitspec.numout = firspec.numout;
	
	% Determine the required output wordlength (the maximum of the required
	% wordlengths for each output branch)
	woutb = zeros(1, firspec.numout);
	for b = 1:firspec.numout
		woutb(b) = arithspec.wdata+ceil(log2(sum(abs(firspec.h{b}))));
	end
	wout = max(woutb);
	
	if genopts.verbose >= 1
		disp(sprintf('Generating FIR filter with %d input branches and %d output branches', firspec.numin, firspec.numout));
		if arithspec.signeddata == 0
			d = 'unsigned';
		else
			d = 'signed';
		end
		disp(sprintf('Using %s data with wordlength %d', d, arithspec.wdata));
	
		if arithspec.signedcoeffs == 0
			c = 'unsigned';
		else
			c = 'signed';
		end
		disp(sprintf('Using %s digit coefficients', c));
	end
	
	for b = 1:firspec.numout
		h = firspec.h{b};
	
		% Generate and choose coefficient representation
		creps = gencoeffreps(h, wout, arithspec.signedcoeffs);
		crep = chooserep(creps, arithspec.ctype, arithspec.cid);
	
		% Compute the needed correction term from the coefficient and data formats
		cterm = calccterm(crep, wout, arithspec.wdata, arithspec.signeddata);
	
		bitspec.branches{b}.h = h;
		bitspec.branches{b}.crep = crep;
		bitspec.branches{b}.cterm = cterm;
		
		if genopts.verbose >= 1
			disp(sprintf('Branch %d impulse response (class %s): [%s]', b-1, crep.class, sprintf(' %d', h)));
			if genopts.verbose >= 2
				for c = 1:size(crep.coeffs, 1)
					disp(sprintf('  coeff %3d: %s', c-1, sprintf(' %2d', crep.coeffs(c, :))));
				end
				disp(sprintf('  corr term: %s', sprintf(' %2d', cterm)));
			end
		end
	end
	
	bitspec.arith = arithspec;
	bitspec.arith.wout = wout;
	
